/*
 * ip_resend.c -- reduce timeout related delays occuring on demand
 *                dialling ppp links with dynamic IP-address assignement
 *
 * Copyright 2000       by Henner Eisen.
 * Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997
 *      The Regents of the University of California.  All rights reserved.
 * Copyright 1998       by Alexey Kuznetsov and Rani Assaf
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef SOCK_DIR_PREFIX
#define SOCK_DIR_PREFIX "/var/ip_resend"
#endif

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <net/if.h>
#include <net/if_packet.h>
#include <net/if_arp.h>
#include <net/ethernet.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <netinet/tcp.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
//#include <stdlib.h>
//#include <time.h>

int debug = 1;
struct timeval tv_start;

/*
 * The following routine is copied from print-ip.c as included in tcpdump-3.4
 *
 * Modified by Henner Eisen 
 * (use Linux/glibc struct iphdr instead of struct ip).
 *
 * As UCB no longer requires to inlcude the advertising clause (3),
 * this is GPL compatible
 */ 

/*
 * Copyright (c) 1988, 1989, 1990, 1991, 1992, 1993, 1994, 1995, 1996, 1997
 *      The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


/*
 * compute an IP header checksum.
 * don't modifiy the packet.
 */
static int
in_cksum(const struct iphdr *ip)
{
        register const u_short *sp = (u_short *)ip;
        register u_int32_t sum = 0;
        register int count;

        /*
         * No need for endian conversions.
         */
        for (count = ip->ihl * 2; --count >= 0; )
                sum += *sp++;
        while (sum > 0xffff)
                sum = (sum & 0xffff) + (sum >> 16);
        sum = ~sum & 0xffff;

        return (sum);
}

/*
 * Replacement for linux kernel's asm coded csum_tcpudp_magic(). Derived
 * from in_cksum and thus also owes credits to in_cksum's authors and UCB.
 */
static u_int16_t csum_tcpudp_magic(u_int32_t saddr,
				   u_int32_t daddr,
				   u_int16_t len,
				   u_int16_t proto,
				   unsigned int sum)
{
	int count;
	struct pseudo_hdr {
		u_int32_t saddr, daddr;
		unsigned char nul, protocol;
		u_int16_t len;
	} ph;
	u_int16_t *sp = (u_int16_t *) &ph;

	ph.saddr = saddr;
	ph.daddr = daddr;
	ph.nul = 0;
	ph.protocol = proto;
	ph.len = len;
	
        /*
         * No need for endian conversions.
         */
	while (sum > 0xffff)
		sum = (sum & 0xffff) + (sum >> 16);
	for (count = sizeof(ph)/2; --count >= 0; )
                sum += *sp++;
        while (sum > 0xffff)
		sum = (sum & 0xffff) + (sum >> 16);
        sum = ~sum & 0xffff;
	
        return (sum);
}

/*
 * Replace source or destination IP address and
 * re-calculate IP and possibly UDP or TCP check sums.
 *
 * tcp/udp address rewriting copied and modified from Linux
 * 2.4.0-test9 kernel net/ipv4/ip_nat_dumb.c by Henner Eisen
 * Original Authors:     Alexey Kuznetsov, Rani Assaf
 */
static int ip_do_nat(unsigned char * buffer, int size, int saddr, int daddr)
{
	struct iphdr * iph = (struct iphdr*) buffer;
        u_int32_t odaddr = iph->daddr;
        u_int32_t osaddr = iph->saddr;
        u_int16_t     check;
        u_int16_t    *cksum;

        /* Rewrite IP header */
        iph->daddr = daddr;
        iph->saddr = saddr;
	iph -> check = 0;
	check = in_cksum(iph);
	iph -> check = check;

        /* If it is the first fragment, rewrite protocol headers */

	if(ntohs(iph->frag_off)&0x1fff) return 0;

	switch(iph->protocol) {
	case IPPROTO_TCP:
		cksum  = (u_int16_t*)&((struct tcphdr*)(((char*)iph) + (iph->ihl<<2)))->check;
		if ((u_int8_t*)(cksum+1) > buffer+size)
			goto truncated;
		check  = csum_tcpudp_magic(iph->saddr, iph->daddr, 0, 0, ~(*cksum));
		*cksum = csum_tcpudp_magic(~osaddr, ~odaddr, 0, 0, ~check);
		break;
	case IPPROTO_UDP:
		cksum  = (u_int16_t*)&((struct udphdr*)(((char*)iph) + (iph->ihl<<2)))->check;
		if ((u_int8_t*)(cksum+1) > buffer+size)
			goto truncated;
		if ((check = *cksum) != 0) {
			check = csum_tcpudp_magic(iph->saddr, iph->daddr, 0, 0, ~check);
			check = csum_tcpudp_magic(~osaddr, ~odaddr, 0, 0, ~check);
			*cksum = check ? : 0xFFFF;
		}
		break;
	default:
		break;
	}

	return 0;
	
truncated:
	return -EINVAL;
}

/*
 * return time elapsed in seconds since t1 as a double
 */
static double tvdiff(struct timeval *t1)
{
	double t;
	struct timeval now[1];

	gettimeofday(now,NULL);
	t  = (now->tv_usec/1000000.);
	t += now->tv_sec;
	t -= (t1->tv_usec/1000000.);
	t -= t1->tv_sec;
	return t;
}


/*
 * convert a double to struct timeval
 */
void dtotv( double t, struct timeval * tv)
{
	if(t<=0.){
		tv->tv_sec = 0;
		tv->tv_usec = 0;
	} else {
		tv->tv_sec  = floor(t);
		tv->tv_usec = floor(1000000 *(t-tv->tv_sec));
		if(tv->tv_usec > 999999) tv->tv_usec = 999999; 
	}
}

/*
 * write hexdump of packet to stdout
 */
static void dump_packet(unsigned char *buffer, int size)
{
	int i;

	for(i=0;i<size;i++){
		printf(" %2.2x", (int) buffer[i]);
	}
}

/*
 * Create and configure a PF_UNIX socket for commincation between
 * ip_resend and ip_resend_wakeup.
 *
 * Returns a socket file descriptor on success, -1 on failure.
 */
static int setup_communication_channel(char *netif, char* path)
{
	struct sockaddr_un addr;
	int fd;

	if( (fd=socket(PF_UNIX,SOCK_DGRAM,0)) < 0){
		perror("error creating PF_LOCAL socket");
		return -1;
	}

	addr.sun_family = AF_LOCAL;
	strncpy(addr.sun_path, path, sizeof(addr.sun_path));
	/*
	 * Terminate possibly concurrent ip_resend tasks communicating
	 * via the same socket.
	 * This should not occur, but certain error scenarios or race
	 * conditions might leave a dangling ip_resend process around.
	 */
	sendto(fd, "term", 5, 0,(struct sockaddr*) &addr, sizeof(addr));
	sendto(fd, "term", 5, 0,(struct sockaddr*) &addr, sizeof(addr));
	/*
	 * Remove possible old socket entry.
	 */
	if(unlink(path)){
		if(errno != ENOENT){
			fprintf(stderr, "ip_resend -c %s: error in unlink(%s)",
				netif, path);
			perror("");
			return -1;
		}
	}
	if(bind(fd,&addr,sizeof(addr))){
		fprintf(stderr, "ip_resend -c %s: error in AF_UNIX bind(%s): ",
		        netif, path);
		perror("");
		return -1;
	}
	
	return fd;
}


/*
 * create and set up a packet socket for snooping outgoing packets.
 *
 * Returns an AF_PACKET socket file descriptor on success, -1 on failure;
 */
static int setup_snoop_socket(struct sockaddr_ll *sl,unsigned char * netif)
{
	int s;
	struct ifreq ifr;

	/*
	 * create a packet socket for snooping outgoing packets
	 */
	if ((s = socket(PF_PACKET, SOCK_DGRAM, htons(ETH_P_IP))) == -1) {
		fprintf(stderr,"ip_resend -s %s: error creating PF_PACKET "
			"socket:",netif);
		perror("");
		return -1;
	}
	memset(&ifr,0,sizeof(ifr));
	strncpy(ifr.ifr_name,netif,sizeof(ifr.ifr_name));
	if(ioctl(s,SIOCGIFINDEX,&ifr)!=0){
		
		fprintf(stderr,"ip_resend -s %s: ioctl(,SIOCGIFINDEX,)",netif);
		perror("");
		return 1;
	}
	if(debug > 1) printf("ip_resend: index of %s is %d\n",
			     netif, ifr.ifr_ifindex);
	memset(sl,0,sizeof(*sl));
	sl->sll_family   = AF_PACKET;
	sl->sll_protocol = htons(ETH_P_ALL);
	sl->sll_ifindex  = ifr.ifr_ifindex;

	if (bind(s, (struct sockaddr *) sl, sizeof(*sl))) {
		fprintf(stderr,"ip_resend -s %s: AF_PACKET bind():", netif);
		perror("");
		return -1;
	}

	return s;
}

/*
 *
 */
static int snoop_first_outgoing_packet(int s, int us, unsigned char *buffer,
				       int bufsize, struct sockaddr_ll * sl,
				       unsigned char * netif)
{
	int size, nfd, fromlen;
	fd_set rfds, efds;
	unsigned char dummy[1500];

	/*
	 * Snoop for first outgoing packet
	 */
	nfd = (us>s ? us : s)+1;
	while(1) {
		FD_ZERO(&rfds);
		FD_SET(us,&rfds);
		FD_SET(s,&rfds);
		FD_ZERO(&efds);
		FD_SET(us,&efds);
		if(select(nfd, &rfds, NULL, &efds, NULL)<0){
			fprintf(stderr, "ip_resend %s:",netif);
			perror("select");
			return -1;
		} else if(FD_ISSET(us,&efds)){
			fprintf(stderr, "ip_resend -s %s: error condition on sesend notification socket",netif);
			return -1;
		} else if(FD_ISSET(s,&efds)){
			fprintf(stderr, "ip_resend -s %s: error condition on snooping packet socket",netif);
			return -1;
		}
		if(!FD_ISSET(s, &rfds) && FD_ISSET(us, &rfds) ){
			/* somebody tried to awake us before an appropriate
			 * outgoing packet was sent */
			if(debug>0) printf("ip_resend -s %s: woken up without outgoing packet beeing present\n",netif);
			/* drain garbage from from socket*/
			recv(us, dummy, sizeof(dummy), 0);
			if(strncmp(dummy,"stop",4)==0) return 0;
			if(strncmp(dummy,"term",4)==0){
				fprintf(stderr, "ip_resend -s %s dangling?: "
					"terminated!\n", netif); 
			} else {
				fprintf(stderr, "ip_resend -s %s: no outgoing "
					"packets: terminated!\n", netif); 
			}
			return -1;
		}

		fromlen = sizeof(*sl);
		if ((size = recvfrom(s, buffer, bufsize, 0,
				     (struct sockaddr*) sl, &fromlen))==-1){
			fprintf(stderr,"ip_resend: -s %s: ", netif);
			perror("recvfrom()");
			return -1;
		}
		if(debug > 1) printf("received packet type %d, proto %d\n", 
				     (int) sl->sll_pkttype,
				     (int) sl->sll_protocol);
		if(sl->sll_pkttype==PACKET_OUTGOING) break;
	};
	return size;
}


static int wait_for_resend_notification(int s, int us, double wake_timeo,
					struct sockaddr_ll *sl,
					struct in_addr *in_new,
					unsigned char * netif,
					unsigned char * path)
{
	int err, nfd, cnt;
	fd_set rfds, efds;
	struct timeval tv;
	struct ifreq ifr;
	unsigned char new_address[81], dummy[1500];

	/*
	 * we are no longer interested in outgoing packets at the snoop device
	 * but are now interested in incoming packets from the outgoing device:
	 */
	memset(&ifr,0,sizeof(ifr));
	strncpy(ifr.ifr_name,netif,sizeof(ifr.ifr_name));
	if(ioctl(s,SIOCGIFINDEX,&ifr)!=0){
		fprintf(stderr,"ip_resend -o %s: ioctl(,SIOCGIFINDEX,)",netif);
		perror("");
		return 1;
	}
	if(debug > 1) printf("ip_resend: index of %s is %d\n",
			     netif, ifr.ifr_ifindex);
	sl->sll_family   = AF_PACKET;
	sl->sll_protocol = htons(ETH_P_IP);
	sl->sll_ifindex  = ifr.ifr_ifindex;
	sl->sll_pkttype  = PACKET_HOST;
	
	if (bind(s, (struct sockaddr *) sl, sizeof(*sl))) {
		fprintf(stderr,"ip_resend -o %s wait_for_resend_notification: bind()",netif);
		perror(":");
		return -1;
	}

	dtotv(wake_timeo,&tv);;

	nfd = (us>s ? us:s) + 1;
	while(1){
		int err;
fd_set rfds, efds;
		FD_ZERO(&rfds);
		FD_SET(us,&rfds);
		FD_SET(s,&rfds);
		FD_ZERO(&efds);
		FD_SET(us,&efds);
		/*
		 * Linux adjusts tv for next wait. If you ever port this to
		 * another OS that does not provide this select() semantics,
		 * dont forget to update tv here.
		 */ 
		err = select(nfd, &rfds, NULL, &efds, &tv);
		if(err==0){
			fprintf(stderr, "ip_resend -o %s: timeout while waiting for re-send notification\n",netif);
			return -1;
		} else if(err<0){
			fprintf(stderr, "ip_resend -o %s %s:",netif,path);
			perror("select");
			return -1;
		}  else if(FD_ISSET(us,&efds)){
			fprintf(stderr, "ip_resend -o %s: error condition on resend notification pipe",netif);
			return -1;
		}
		if(FD_ISSET(s, &rfds)){
			/* we drain garbage from the socket that could
			   accumulate while waiting for re-send notification */
			int fromlen = sizeof(sl);
			recvfrom(s, dummy, sizeof(dummy), 0,
				 (struct sockaddr*) &sl, &fromlen);
			if(debug>0) fprintf(stderr,"@");
		}
		if(FD_ISSET(us, &rfds)) break; 
	}

	/*
	 * read new IP address of network interface from the named pipe
	 */
	cnt = recv(us,new_address,sizeof(new_address)-1,0);
	if( cnt < 0 ){
		fprintf(stderr, "ip_resend -o %s: recv error from ", netif); 
		perror(path);
		return -1;
	}
	new_address[cnt]=0;
	if(new_address[cnt-1]=='\n') new_address[cnt-1]=0; 
	if(debug>1) printf("ip_resend -o %s: received new address %s\n",
			   netif,new_address);
	if(strcmp(new_address,"stop")==0) return 1;
	if(strcmp(new_address,"term")==0){
		fprintf(stderr, "ip_resend -o %s dangling?: terminated!\n",
			netif); 
		return -1;
	}
	if(inet_aton(new_address,in_new)==0){
		fprintf(stderr, "ip_resend -o %s: '%s' is not a valid "
			"inet address\n", netif, new_address);
		return -1;
	}
	return 0;
}


static int wait_for_response(int s, unsigned char *buffer, int bufsize,
			     double resp_timeo, struct sockaddr_ll *sl,
			     u_int32_t from_addr, u_int32_t to_addr,
			     unsigned char * netif, unsigned char *path)
{
	int err, fromlen, size;
	fd_set rfds, efds;
	struct timeval tv;
	struct iphdr * iph = (struct iphdr *) buffer;

	dtotv(resp_timeo,&tv);

	if(debug > 0) printf("ip_resend -o %s: awaiting response packet within"
			     " %.3f seconds\n", netif, resp_timeo); 
	while(1) {
		/*
		 * wait for incoming response packet
		 */
		FD_ZERO(&rfds);
		FD_SET(s,&rfds);
		FD_ZERO(&efds);
		FD_SET(s,&efds);
		/*
		 * Linux adjusts tv for next wait. If you ever port this to
		 * another OS that does not provide this select() semantics,
		 * dont forget to update tv here.
		 */ 
		err = select(s+1, &rfds, NULL, &efds, &tv);
		if(err==0){
			fprintf(stderr, "ip_resend -o %s: timeout while waiting for response packet\n",netif);
			return -1;
		} else if(err<0){
			fprintf(stderr, "ip_resend -o %s -c %s:",netif,path);
			perror("select");
			return -1;
		}  else if(FD_ISSET(s,&efds)){
			fprintf(stderr, "ip_resend -o %s: error condition on response socket\n",netif);
			return -1;
		}

		fromlen = sizeof(*sl);
		if ((size = recvfrom(s, buffer, bufsize, 0,
				     (struct sockaddr*) sl, &fromlen))==-1){
			fprintf(stderr,"ip_resend: %s: ", netif);
			perror("recvfrom()");
			return -1;
		}
		if(debug>0) fprintf(stderr,"#");
		if(debug > 1) printf("received packet type %d, proto %d\n", 
				     (int) sl->sll_pkttype,
				     (int) sl->sll_protocol);
		if(sl->sll_pkttype  != PACKET_HOST) continue;
		if(sl->sll_protocol != htons(ETH_P_IP)) continue;
		
		if( (iph->daddr!=to_addr) || (iph->saddr!=from_addr) ){
			if(debug>0) printf("ip_resend -o %s: response packet from %s to %s after %.3fs: addresses don't match re-sent packet\n", netif, inet_ntoa(iph->saddr), inet_ntoa(iph->daddr),tvdiff(&tv_start));
			continue;
		}
		break;
	}
	return size;
}


static int loopback_response_packet(int s, unsigned char *buffer, int size,
				    unsigned char * netif)
{
	struct ifreq ifr;
	struct sockaddr_ll sl;

	memset(&ifr,0,sizeof(ifr));
	strcpy(ifr.ifr_name,"lo");
	if(ioctl(s,SIOCGIFINDEX,&ifr)!=0){
		perror("ip_resend: ioctl(,SIOCGIFINDEX,)");
		return -1;
	}
	if(debug>1) printf("ip_resend: index of lo is %d\n",ifr.ifr_ifindex);
	if(debug>1) { dump_packet(buffer,size); printf("\n"); }; 
	/*
	 * re-insert modified response packet in Linux's net
	 * input path via lo (local loopback) network interface 
	 */ 
	memset(&sl,0,sizeof(sl));
	sl.sll_family   = AF_PACKET;
	sl.sll_ifindex  = ifr.ifr_ifindex;
	sl.sll_protocol = ntohs(ETH_P_IP);
	sl.sll_pkttype  = PACKET_HOST;
#if 0 /* not necessary */
	if (bind(s, (struct sockaddr *) &sl, sizeof(sl))) {
		fprintf(stderr,"ip_resend -o %s: loopback: bind()",netif);
		perror(":");
		return 1;
	}
#endif
	if(debug > 1) printf("ip_resend -o %s: re-input translated response packet via lo\n", netif); 
	if (sendto(s,buffer,size,0,(struct sockaddr*) &sl, sizeof(sl)) == -1){
		fprintf(stderr,"ip_resend -o %s: loopback sendto()",netif);
		perror(":");
		return -1;
	}
	return 0;
}

static void pr_usage_and_exit_1(void)
{
	fprintf(stderr,
		"usage: ip_resend  -o OUTGOING_INTERFACE [-s SNOOP_INTERFACE][-c COMMUNICATION_SOCKET] [-w WAKEUP_WAIT_MAX] [-{r|R} RESPONSE_WAIT_MAX] [-d DEBUG_LEVEL]");
	exit(1);
}

int main(int argc, char **argv)
{
	unsigned char buffer[1500], path_buf[PATH_MAX];
	unsigned int csum, proto, ignore=0;
	int s, fromlen, cnt, i, j, us, nfd, err, size, new_ttl,
		dont_translate_response=0, c;
	struct sockaddr_ll sl_bind, sl_from, sl;
	struct in_addr in_new, in_s, in_d;
	struct iphdr * iph = (struct iphdr *) buffer;
	struct timeval tv;
	double td, snoop_wait, wake_timeo=10., resp_timeo=10., resend_delay=0.;

	unsigned char *path=NULL, *out_dev=NULL, *snoop_dev=NULL; 
	
	while( (c=getopt(argc,argv,"o:s:c:w:r:R:d:M")) != EOF ){
		switch(c){
		case 'o':
			out_dev = optarg;
			break;
		case 's':
			snoop_dev = optarg;
			break;
		case 'c':
			path = optarg;
			break;
		case 'w':
			sscanf(optarg,"%lf",&wake_timeo);
			break;
		case 'R':
			dont_translate_response=1;
		case 'r':
			sscanf(optarg,"%lf",&resp_timeo);
			break;
		case 'd':
			debug=atoi(optarg);
			break;
		case 'M':
			/* currently do nothing */
			break;
		default:
			pr_usage_and_exit_1();
		}
	}
	
	if(! out_dev) pr_usage_and_exit_1();
	if(! snoop_dev) snoop_dev = out_dev;
	if(! path){
		int n;
		n=snprintf(path_buf,sizeof(path_buf),
			   SOCK_DIR_PREFIX "/%s", out_dev);
		if((n<0) || n>=(sizeof(path_buf)-1)){
			fprintf(stderr, "ip_resend -o %s: option too long!\n");
			return 1;
		}
		path = path_buf;
	}

	if((us=setup_communication_channel(out_dev,path)) < 0) return 1;

	if((s=setup_snoop_socket(&sl_bind,snoop_dev))<0){
		unlink(path);
		return 1;
	};
	
	if(debug > 0)
		printf("ip_resend -s %s: waiting for outgoing packet\n", snoop_dev);
	size=snoop_first_outgoing_packet(s,us,buffer,sizeof(buffer),
					 &sl_from, snoop_dev);
	if(size<=0){
		unlink(path);
		return size==0 ? 0:1;
	}

	gettimeofday(&tv_start,NULL);
	if(sl_from.sll_protocol != htons(ETH_P_IP)){
		if(debug>0) printf("ipresend %s: First outgoing packet on %s is not IP\n",out_dev,snoop_dev);
		return 0;
	}

	in_s.s_addr = iph -> saddr;
	in_d.s_addr = iph -> daddr;
	proto = iph->protocol;
	if(debug > 0) printf("ip_resend -s %s: outgoing IP packet from %s to "
			     , snoop_dev, inet_ntoa(in_s));
	if(debug > 0) printf("%s, proto %d: ",
			      inet_ntoa(in_d), (int) iph->protocol); 
	if(debug > 1) dump_packet(buffer,size);
	if(debug > 0) printf("\n");

	if(debug>0)printf("ip_resend -o %s: awaiting re-send notification from %s within %.3f seconds.\n",
			  out_dev, path, wake_timeo);
	err = wait_for_resend_notification(s,us,wake_timeo,&sl_bind,&in_new,
					   out_dev,path);
	unlink(path);
	if(err<0)return 1;
	if(err>0)return 0;

	if(debug > 1){
		printf("old csum is %hu\n", iph -> check);
		printf("ip-version %d len %d ",
		       (int)iph -> version, (int)iph -> ihl);
	}
	/*
	 * Maybe delay resending by specified number of milliseconds
	 */
	if( resend_delay > 0.){
		dtotv(resend_delay,&tv);
		td = tvdiff(&tv_start);
		printf("ip_resend -o %s: delaying resend at %.3fs for %.3fs\n",
		       out_dev,td,resend_delay);
		select(0,NULL,NULL,NULL,&tv);
	}
	/*
	 * Adjust source IP address of snooped packet according to new
	 * interface address and re-calculate ttl and check sums.
	 */
	td = tvdiff(&tv_start);
	new_ttl = iph->ttl - floor(td);
	if(new_ttl <=0){
		if(debug>0)printf("ip_resend -o %s: ttl %d expired after "
				  "%.3fs\n", out_dev, (int) iph->ttl, td);
		return 0;
	}
	iph->ttl = new_ttl;
	if(ip_do_nat(buffer, sizeof(buffer), in_new.s_addr, iph->daddr)){
		perror("ip_do_nat()");
		return 1;
	}
	if(debug > 1) printf("ip_resend -o %s: new csum is %hu\n",
			     out_dev, iph -> check);
	if(debug > 1){ dump_packet(buffer,size); printf("\n");}

	/*
	 * resend the snooped packet with adjusted source IP address
	 */
	if(debug>0) printf("ip_resend -o %s: re-sending with new source address %s after %.3fs\n", out_dev, inet_ntoa(in_new), td);
	sl = sl_from;
	sl.sll_pkttype=PACKET_OUTGOING;
	if (sendto(s,buffer,size,0,(struct sockaddr*) &sl, sizeof(sl)) == -1){
		perror("ip_resend: sendto()");
		return 1;
	}
	
	if( resp_timeo <= 0. ) return 0;

	/*
	 * await response for re-sent packet
	 */ 
	size=wait_for_response(s,buffer,sizeof(buffer),resp_timeo, &sl_from,
			       in_d.s_addr, in_new.s_addr, out_dev, path);
	if(size < 0) return 1;
	td = tvdiff(&tv_start);
	if(debug > 0) printf("ip_resend -o %s: response from %s to ", out_dev,
			     inet_ntoa(iph->saddr));
	if(debug > 0) printf("%s, proto %d after %.3fs:",
			     inet_ntoa(iph->daddr), (int)iph->protocol, td); 
	if(debug > 1) dump_packet(buffer,size);
	if(debug > 0) printf("\n");

	if(dont_translate_response) return 0;
	/*
	 * replace destination address of response packet by old
	 * interface/source address and re-calculate IP header check sums
	 */
	if(ip_do_nat(buffer, sizeof(buffer), iph->saddr, in_s.s_addr)){
		perror("ip_do_nat() response");
		return 1;
	}
	if(loopback_response_packet(s,buffer,size,out_dev)) return 1;
	return 0;
}
